var globalBuffer = [];
var tabDataMap = {};

var closeTab = function(tabId) {
  if (tabDataMap[tabId] !== undefined) {
    tabDataMap[tabId]['playContext'].close();
    tabDataMap[tabId]['recordContext'].close();
    tabDataMap[tabId]['streamMedia'].getTracks().forEach(track => track.stop());
    delete tabDataMap[tabId];
  }
};

chrome.runtime.onMessage.addListener(async (message) => {
  if (message.target === 'offscreen') {
    switch (message.type) {
      case 'start-recording':
        startRecording(message.data, message.tabId);
        break;
      case 'stop-recording':
        closeTab(message.tabId);
        stopRecording(message.tabId);
        break;
      default:
        throw new Error('Unrecognized message:', message.type);
    }
  }

  if (message.closeTab) {
    closeTab(message.closeTab);
  }
});

let recorder;
let data = [];

async function startRecording(streamId, tabId) {

    if (recorder?.state === 'recording') {
      throw new Error('Called startRecording while recording is in progress.');
    }

    tabDataMap[tabId] = {
      streamMedia: null,
      playContext: null,
      recordContext: null
    };

    tabDataMap[tabId]['streamMedia'] = await navigator.mediaDevices.getUserMedia({
      audio: {
        mandatory: {
          chromeMediaSource: 'tab',
          chromeMediaSourceId: streamId
        }
      }
    });

    // Continue to play the captured audio to the user.
    tabDataMap[tabId]['playContext'] = new AudioContext();
    const source = tabDataMap[tabId]['playContext'].createMediaStreamSource(tabDataMap[tabId]['streamMedia']);
    source.connect(tabDataMap[tabId]['playContext'].destination);


    tabDataMap[tabId]['recordContext'] = new (window.AudioContext || window.webkitAudioContext)({
      sampleRate: 16000 // 设置目标采样率为16kHz
    });

    const mediaStreamSource = tabDataMap[tabId]['recordContext'].createMediaStreamSource(tabDataMap[tabId]['streamMedia']);

    const numberOfInputChannels = 1; // 单声道输入
    const numberOfOutputChannels = 1; // 单声道输出

    let bufferSize = 1024;
    let maxBufferSize = 10;

    const scriptNode = tabDataMap[tabId]['recordContext'].createScriptProcessor(bufferSize, numberOfInputChannels, numberOfOutputChannels);

    scriptNode.onaudioprocess = function(audioProcessingEvent) {                    

        const inputBuffer = audioProcessingEvent.inputBuffer;
        const inputData = inputBuffer.getChannelData(0); // 获取单声道输入数据
        let samplesCopy = inputData.slice(0);

        if (globalBuffer.length > bufferSize * maxBufferSize) {
            globalBuffer = globalBuffer.slice(0 - bufferSize * maxBufferSize);
        }

        for (let i = 0; i < samplesCopy.length; i++) {
            globalBuffer.push(samplesCopy[i]);
        }

        let maxOutputSize = 320;
        let length = globalBuffer.length;
        if (length < maxOutputSize) {
            return;
        }

        let size = parseInt(length / maxOutputSize);
        for (var sizeIndex = 0; sizeIndex < size; sizeIndex++) {
            var output = new ArrayBuffer(maxOutputSize * 2);
            var view = new DataView(output);
            var offset = 0;

            var sum = 0;
            for (var index = sizeIndex * maxOutputSize; index < sizeIndex * maxOutputSize + maxOutputSize; index++) {
                var s = Math.max(-1, Math.min(1, globalBuffer[index]))
                let int16Val = s < 0 ? s * 0x8000 : s * 0x7FFF;
                sum += Math.abs(int16Val);
                view.setInt16(offset, int16Val, true);
                offset += 2;
            }

            try {
              chrome.runtime.sendMessage({
                pcmToWorker: true,
                data: Array.from(new Int16Array(output)),
                tabId: tabId
              });
            } catch {
              closeTab(tabId);
            }
        }

        var newBuffer = [];
        var offset = 0;
        for (var index = size * maxOutputSize; index < globalBuffer.length; index++) {
            newBuffer[offset++] = globalBuffer[index];
        }
        globalBuffer = newBuffer;
    };

    mediaStreamSource.connect(scriptNode);
    scriptNode.connect(tabDataMap[tabId]['recordContext'].destination); 
}

async function stopRecording(tabId) {
  try {
    chrome.tabs.sendMessage(tabId, { stopTab: tabId });
  } catch (e) {}
}
